# $Id: mkgraphs.R,v 1.3 2011/10/17 21:25:48 srevilak Exp $

rm(list=ls())
require(lattice);

my.read <- function(filename) {
  d <- read.table(filename, header=TRUE, sep="\t");
  d;
}


##
## Function to print a trellis object to the screen, and then to a
## .pdf
##
my.print <- function(x, outfile) {
  print(x);
  dev.copy(pdf, file=outfile,
           pointsize=12,
           width=8, height=5,
           useDingbats=FALSE)
  dev.off();
}


##
## Generate a custom strip for xyplot graphs
##
my.strip.custom <- function(factor.levels, bg, shingle.intervals, ...) {
    clean.levels <- gsub("  *\\}", "", factor.levels);
    clean.levels <- gsub("\\{  *", "", clean.levels);
    strip.default(factor.levels=paste(clean.levels, "-row hotspot", sep=""),
                  bg="gray85",
                  shingle.intervals=NULL, ...);
}


##
## Generic wrapper around xyplot.  Returns the trellis object
##
my.xyplot <- function(formula, data, title, ylab, pch, legend, ...) {
  g <- xyplot(formula,
              data=data,
              as.table=TRUE,
              col="black",
              type="b",
              pch=pch,
              cex=my.cex,
              main=title,
              ylab=ylab,
              strip=my.strip.custom,
              par.settings=my.par.settings,
              panel=function(...) {
                panel.grid(col="gray50", lty=2, h=-1, v=-1)
                panel.xyplot(...)
              },
              key=list(columns=length(legend),
                space="top",
                text=list(legend),
                lines=list(lty=1, type="b", pch=pch),
                padding=10,
                col="black"
                ),
              ...
              );
  g;
}


##################################################################

###
### Generate CTPS graphs for configuration "xs" (ie, s5u1)
###
plot.ctps <- function(xs) {

  sub.data <- subset(all.data,
                     skun == xs &
                     (hotspot == 800 |
                      hotspot == 1200))[c('MPL', 'hotspot',
                        'isolation.level', 'ctps')]
                               
  sub.data <- reshape(sub.data,
                      idvar=c("MPL", 'hotspot'),
                      timevar="isolation.level",
                      direction="wide");

  names(sub.data) <- gsub("ctps.", "", names(sub.data));

  x <- my.xyplot(formula=S2PL + SI + ESSI + PSSI ~ MPL | hotspot,
                 data=sub.data,
                 title=sprintf("%s CTPS, Varying Hotspot Sizes", xs),
                 ylab="CTPS",
                 pch=1:4,
                 legend=c('S2PL', 'SI', 'ESSI', 'PSSI'));

  my.print(x, sprintf("img-05-%s-ctps.pdf", xs));
}



##################################################################


##
## Plot FUW errors for a specific configuration (xs)
##
plot.fuw <- function(xs) {
  sub.data <- subset(all.data,
                     skun == xs &
                     isolation.level != 'S2PL' &
                     (hotspot == 800 |
                      hotspot == 1200))[
                                        c('MPL',
                                          'hotspot',
                                          'isolation.level',
                                          'pct_fuw')
                                        ]

  sub.data <- reshape(sub.data,
                      idvar=c("MPL", 'hotspot'),
                      timevar="isolation.level",
                      direction="wide");

  names(sub.data) <- gsub("pct_fuw.", "", names(sub.data));

  x <- my.xyplot(formula=SI + ESSI + PSSI ~ MPL | hotspot,
                 data=sub.data,
                 title=sprintf("%s %%First Updater Wins Aborts", xs),
                 ylab="%Aborted Transactions (FUW)",
                 pch=2:4,
                 legend=c('SI', 'ESSI', 'PSSI'));

  my.print(x, sprintf("img-05-%s-fuw.pdf", xs));
}


##
## plot serialization errors for a specific configuration
##
plot.sr.err <- function(xs) {

  measure <- "pct_sr_err";
  
  sub.data <- subset(all.data,
                     skun == xs &
                     isolation.level != 'S2PL' &
                     isolation.level != 'SI' & 
                     (hotspot == 800 |
                      hotspot == 1200))[
                                        c('MPL',
                                          'hotspot',
                                          'isolation.level',
                                          measure)
                                        ]

  sub.data <- reshape(sub.data,
                      idvar=c("MPL", 'hotspot'),
                      timevar="isolation.level",
                      direction="wide");

  names(sub.data) <- gsub(paste(measure, ".", sep=""), "",
                          names(sub.data));

  x <- my.xyplot(formula=ESSI + PSSI ~ MPL | hotspot,
                 data=sub.data,
                 title=sprintf("%s %%Serialization Aborts", xs),
                 ylab="%Aborted Transactions (non-SR)",
                 pch=3:4,
                 legend=c('ESSI', 'PSSI'));

  my.print(x, sprintf("img-05-%s-sr-err.pdf", xs));

}


##
## Plot transaction duration
##
plot.tx.duration <- function(xs, ylim=NULL) {

  measure <- "avg.commited.tx.duration";
  
  sub.data <- subset(all.data,
                     skun == xs &
                     hotspot > 400)[
                                    c('MPL',
                                      'hotspot',
                                      'isolation.level',
                                      measure)
                                    ]

  sub.data <- reshape(sub.data,
                      idvar=c("MPL", 'hotspot'),
                      timevar="isolation.level",
                      direction="wide");

  names(sub.data) <- gsub(paste(measure, ".", sep=""), "",
                          names(sub.data));

  x <- my.xyplot(formula=S2PL + SI + ESSI + PSSI ~ MPL | hotspot,
                 data=sub.data,
                 title=sprintf("%s Avg. Duration of Committed Transactions", xs),
                 ylab="Duration (ms)",
                 pch=1:4,
                 legend=c('S2PL', 'SI', 'ESSI', 'PSSI'),
                 ylim=ylim);

  my.print(x, sprintf("img-05-%s-tx-duration.pdf", xs));

}


##
## Plot ESSI, PSSI CTG sizes for s3u1, s5u1
##
plot.ctg.size <- function(xs) {
  measure <- "avg.ctg.size";

  sub.data <- subset(all.data,
                     (isolation.level == "ESSI" |
                      isolation.level == "PSSI") &
                     (skun == xs) &
                     (hotspot == 800 |
                      hotspot == 1200))[
                        c('MPL',
                          'hotspot',
                          'isolation.level',
                          measure)
                        ]
  
  sub.data <- reshape(sub.data,
                      idvar=c("MPL", 'hotspot'),
                      timevar="isolation.level",
                      direction="wide");

  names(sub.data) <- gsub(paste(measure, ".", sep=""), "", names(sub.data));

  x <- my.xyplot(formula= ESSI + PSSI ~ MPL
                 | hotspot,
                 data=sub.data,
                 title=sprintf("%s Average CTG Size", xs),
                 ylab="CTG Size (# nodes)",
                 pch=3:4,
                 legend=c('ESSI', 'PSSI'));

  my.print(x, sprintf("img-05-%s-ctg-size.pdf", xs));

}



##
## Plot # of edges traversed during cycle testing
##
plot.ct.edges <- function(xs) {
  measure <- "avg.cycle.edges.traversed";

  sub.data <- subset(all.data,
                     (isolation.level == "PSSI") &
                     (skun == xs))[
                        c('MPL',
                          'hotspot',
                          'isolation.level',
                          measure)
                        ]
  
  sub.data <- reshape(sub.data,
                      idvar=c("MPL", 'hotspot'),
                      timevar="isolation.level",
                      direction="wide");

  names(sub.data) <- gsub(paste(measure, ".", sep=""), "", names(sub.data));

  x <- my.xyplot(formula= PSSI ~ MPL
                 | hotspot,
                 data=sub.data,
                 title=sprintf("%s Edges Traversed During Cycle Testing", xs),
                 ylab="#Edges Traversed",
                 pch=4,
                 legend=c('PSSI'));

  my.print(x, sprintf("img-05-%s-ct-edges.pdf", xs));
}


##
## Plot avg cycle length
##
plot.cycle.length <- function(xs) {
  measure <- "avg.cycle.length";

  sub.data <- subset(all.data,
                     (isolation.level == "PSSI") &
                     (skun == xs))[
                        c('MPL',
                          'hotspot',
                          'isolation.level',
                          measure)
                        ]
  
  sub.data <- reshape(sub.data,
                      idvar=c("MPL", 'hotspot'),
                      timevar="isolation.level",
                      direction="wide");

  names(sub.data) <- gsub(paste(measure, ".", sep=""), "", names(sub.data));

  x <- my.xyplot(formula= PSSI ~ MPL
                 | hotspot,
                 data=sub.data,
                 title=sprintf("%s Average Cycle Length", xs),
                 ylab="Avg. Cycle length",
                 pch=4,
                 legend=c('PSSI'));

  my.print(x, sprintf("img-05-%s-cycle-len.pdf", xs));
}



##
## Plot transaction abort rates
##
plot.pct.failed <- function(xs) {

  measure <- "pct_failed";
  
  sub.data <- subset(all.data,
                     skun == xs &
                     hotspot > 400)[
                                    c('MPL',
                                      'hotspot',
                                      'isolation.level',
                                      measure)
                                    ]

  sub.data <- reshape(sub.data,
                      idvar=c("MPL", 'hotspot'),
                      timevar="isolation.level",
                      direction="wide");

  names(sub.data) <- gsub(paste(measure, ".", sep=""), "",
                          names(sub.data));

  x <- my.xyplot(formula=S2PL + SI + ESSI + PSSI ~ MPL | hotspot,
                 data=sub.data,
                 title=sprintf("%s Transaction Abort Rates", xs),
                 ylab="%Aborted Transactions (any reason)",
                 pch=1:4,
                 legend=c('S2PL', 'SI', 'ESSI', 'PSSI'));

  my.print(x, sprintf("img-05-%s-pct-failed.pdf", xs));
}




##################################################################

##
## Regenerate all of the graphs
##
all.plots <- function() {

  # CTPS graphs
  plot.ctps("s5u1");

  # error rates  
  plot.fuw("s5u1");
  plot.sr.err("s5u1");

  # combined error rates
  plot.pct.failed("s5u1");
  
  # transaction durations
  plot.tx.duration("s5u1", ylim=c(0, 35));

  # ctg sizes
  plot.ctg.size("s5u1");
}

all.data <- my.read("../dissertation/all-data.csv");

# trellis.par.get()

##
## xyplot size parameters
##
my.cex <- 1.2
my.cex.main <- 4
my.par.settings=list(
  par.main.text=list(cex=1.5),
  par.xlab.text=list(cex=1.2),
  par.ylab.text=list(cex=1.2),
  axis.text=list(cex=1.2),
  add.text=list(cex=1.0)         
  );

options(scipen=5)

